#!/bin/bash

# Linux Dependencies Installation Script
# This script installs all required dependencies for the Al Mirag Poultry Management app

set -e

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_status "Installing dependencies for Al Mirag Poultry Management..."

# Detect Linux distribution
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS=$NAME
    VER=$VERSION_ID
else
    print_error "Cannot detect Linux distribution"
    exit 1
fi

print_status "Detected OS: $OS $VER"

# Update package lists
print_status "Updating package lists..."
if command -v apt-get &> /dev/null; then
    sudo apt-get update
elif command -v yum &> /dev/null; then
    sudo yum update -y
elif command -v dnf &> /dev/null; then
    sudo dnf update -y
elif command -v pacman &> /dev/null; then
    sudo pacman -Sy
else
    print_warning "Package manager not detected. Please install dependencies manually."
fi

# Install Python 3 and pip
print_status "Installing Python 3 and pip..."
if command -v apt-get &> /dev/null; then
    sudo apt-get install -y python3 python3-pip python3-venv python3-dev
elif command -v yum &> /dev/null; then
    sudo yum install -y python3 python3-pip python3-devel
elif command -v dnf &> /dev/null; then
    sudo dnf install -y python3 python3-pip python3-devel
elif command -v pacman &> /dev/null; then
    sudo pacman -S --noconfirm python python-pip
else
    print_warning "Please install Python 3 and pip manually"
fi

# Install system dependencies
print_status "Installing system dependencies..."
if command -v apt-get &> /dev/null; then
    sudo apt-get install -y build-essential libssl-dev libffi-dev libsqlite3-dev
elif command -v yum &> /dev/null; then
    sudo yum install -y gcc openssl-devel libffi-devel sqlite-devel
elif command -v dnf &> /dev/null; then
    sudo dnf install -y gcc openssl-devel libffi-devel sqlite-devel
elif command -v pacman &> /dev/null; then
    sudo pacman -S --noconfirm base-devel openssl libffi sqlite
fi

# Install Apache and mod_wsgi (optional)
print_status "Installing Apache and mod_wsgi..."
if command -v apt-get &> /dev/null; then
    sudo apt-get install -y apache2 libapache2-mod-wsgi-py3
    sudo a2enmod wsgi
    sudo a2enmod rewrite
    sudo a2enmod headers
    sudo a2enmod expires
    sudo a2enmod deflate
elif command -v yum &> /dev/null; then
    sudo yum install -y httpd mod_wsgi
    sudo systemctl enable httpd
elif command -v dnf &> /dev/null; then
    sudo dnf install -y httpd mod_wsgi
    sudo systemctl enable httpd
elif command -v pacman &> /dev/null; then
    sudo pacman -S --noconfirm apache mod_wsgi
    sudo systemctl enable httpd
fi

# Install Python dependencies
print_status "Installing Python dependencies..."

# Check if we're in a virtual environment
if [[ "$VIRTUAL_ENV" != "" ]]; then
    print_status "Using virtual environment: $VIRTUAL_ENV"
    PIP_CMD="pip"
else
    print_warning "Not in a virtual environment. Consider creating one:"
    print_warning "python3 -m venv venv && source venv/bin/activate"
    
    if command -v pip3 &> /dev/null; then
        PIP_CMD="pip3"
    elif command -v pip &> /dev/null; then
        PIP_CMD="pip"
    else
        print_error "pip is not installed"
        exit 1
    fi
fi

# Upgrade pip
print_status "Upgrading pip..."
$PIP_CMD install --upgrade pip

# Install requirements
if [ -f "requirements.txt" ]; then
    print_status "Installing requirements from requirements.txt..."
    $PIP_CMD install -r requirements.txt
else
    print_status "Installing core dependencies..."
    $PIP_CMD install Flask Flask-Login Flask-WTF Flask-SQLAlchemy WTForms email-validator
fi

# Install additional useful packages
print_status "Installing additional packages..."
$PIP_CMD install gunicorn psutil

print_success "Dependencies installed successfully!"

# Create virtual environment recommendation
if [[ "$VIRTUAL_ENV" == "" ]]; then
    print_warning "Consider creating a virtual environment for better isolation:"
    echo "python3 -m venv venv"
    echo "source venv/bin/activate"
    echo "pip install -r requirements.txt"
fi

# Test installation
print_status "Testing installation..."
python3 -c "
import flask
import flask_login
import flask_wtf
import flask_sqlalchemy
import wtforms
print('All dependencies imported successfully!')
"

print_success "🎉 Installation completed successfully!"
print_status "Next steps:"
echo "1. Run the application: ./start_linux.sh"
echo "2. Or deploy with: ./deploy_linux.sh"
echo "3. Configure your web server to use passenger_wsgi.py"
